// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright (c) 2022 Amazon.com, Inc. or its affiliates

#include <linux/gpio/consumer.h>
#include <linux/gpio/driver.h>
#include <linux/module.h>
#include <linux/netlink.h>
#include <linux/of.h>
#include <linux/platform_device.h>

#include <net/sock.h>

#define GPIO_RED__GROUP 31

struct gpio_red_priv {
	struct gpio_chip	gc;
	struct device		*dev;
	struct sock		*sock;
	struct work_struct	work;
	int			value;
};

static void gpio_red__sync(struct work_struct *work)
{
	struct gpio_red_priv	*priv;
	struct sk_buff		*skb;
	struct nlmsghdr		*nlh;
	int			*state;
	int			ret;

	priv = container_of(work, struct gpio_red_priv, work);

	skb = nlmsg_new(sizeof(int), GFP_KERNEL);
	if (!skb) {
		dev_err(priv->dev, "nlmsg_new() failed");

		return;
	}

	nlh = nlmsg_put(skb, 0, 0, NLMSG_DONE, sizeof(priv->value), 0);
	state = nlmsg_data(nlh);
	*state = priv->value;

	ret = nlmsg_multicast(priv->sock, skb, 0, GPIO_RED__GROUP, GFP_KERNEL);
	if (ret)
		dev_err(priv->dev, "nlmsg_multicast() failed %d", ret);
}

static int gpio_red__get_value(struct gpio_chip *gc, unsigned offset)
{
	struct gpio_red_priv *priv;

	priv = container_of(gc, struct gpio_red_priv, gc);

	dev_dbg(priv->dev, "%s(offset = %d)", __func__, offset);

	return priv->value;
}

static void gpio_red__set_value(struct gpio_chip *gc, unsigned offset, int val)
{
	struct gpio_red_priv *priv;

	priv = container_of(gc, struct gpio_red_priv, gc);
	priv->value = val;

	dev_dbg(priv->dev, "%s(offset = %d, val = %d)", __func__, offset, val);

	schedule_work(&priv->work);
}

static int gpio_red__direction_output(struct gpio_chip *gc, unsigned offset, int val)
{
	struct gpio_red_priv *priv;

	priv = container_of(gc, struct gpio_red_priv, gc);
	priv->value = val;

	dev_dbg(priv->dev, "%s(offset = %d, val = %d)", __func__, offset, val);

	schedule_work(&priv->work);

	return 0;
}

static int gpio_red__direction_input(struct gpio_chip *gc, unsigned offset)
{
	struct gpio_red_priv *priv;

	priv = container_of(gc, struct gpio_red_priv, gc);

	dev_dbg(priv->dev, "%s(offset = %d)", __func__, offset);

	return 0;
}

static int gpio_red__probe(struct platform_device *pdev)
{
	int			ret;
	struct gpio_red_priv	*priv;

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->dev = &pdev->dev;
	platform_set_drvdata(pdev, priv);

	priv->sock = netlink_kernel_create(&init_net, NETLINK_USERSOCK, NULL);
	if (!priv->sock)
		return -EIO;

	INIT_WORK(&priv->work, gpio_red__sync);
	priv->gc.label = "gpio-red";
	priv->gc.direction_output = gpio_red__direction_output;
	priv->gc.direction_input = gpio_red__direction_input;
	priv->gc.get = gpio_red__get_value;
	priv->gc.set = gpio_red__set_value;
	priv->gc.base = -1;
	priv->gc.ngpio = 1;
	priv->gc.parent = &pdev->dev;
	priv->gc.owner = THIS_MODULE;
	ret = gpiochip_add(&priv->gc);
	if (ret)
		netlink_kernel_release(priv->sock);

	return ret;
}

static int gpio_red__remove(struct platform_device *pdev)
{
	struct gpio_red_priv *priv = platform_get_drvdata(pdev);

	gpiochip_remove(&priv->gc);
	cancel_work_sync(&priv->work);
	netlink_kernel_release(priv->sock);

	return 0;
}

static const struct of_device_id gpio_red_match_table[] = {
	{ .compatible = "goldfish,gpio-red", },
	{},
};
MODULE_DEVICE_TABLE(of, gpio_red_match_table);

static struct platform_driver gpio_red_driver = {
	.probe		= gpio_red__probe,
	.remove		= gpio_red__remove,
	.driver		= {
		.name	= "gpio_red",
		.of_match_table = gpio_red_match_table,
	},
};
module_platform_driver(gpio_red_driver);

MODULE_AUTHOR("Oleh Kravchenko <o.kravchenko@ringteam.com>");
MODULE_DESCRIPTION("GPIO expander driver for RED");
MODULE_LICENSE("GPL");
